<?php
/* --------------------------------------------------------------
 PluginRegistration.php 2020-04-21
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Application\Kernel\Bootstrapper;

use Gambio\Core\Application\Application;
use Gambio\Core\Application\Kernel\AbstractBootstrapper;
use Gambio\Core\Application\Plugins\Abstraction\PluginServiceProvider;
use Gambio\Core\Application\Plugins\Plugin;
use Gambio\Core\Application\Plugins\PluginRegistryService;
use Gambio\Core\Command\Interfaces\CommandHandlerProvider;
use Gambio\Core\Event\EventListenerProvider;
use Gambio\Core\Logging\LoggerBuilder;
use Psr\Log\LoggerInterface;
use Slim\App as SlimApp;
use Throwable;
use function class_exists;
use function file_exists;
use function implode;

/**
 * Class PluginRegistration
 * @package Gambio\Core\Application\Kernel\Bootstrapper
 */
class PluginRegistration extends AbstractBootstrapper
{
    /**
     * @inheritDoc
     */
    public function boot(Application $application): void
    {
        $service = $this->getPluginsService($application);
        
        if (!$service->exists()) {
            $service->rebuild();
        }
        
        $pluginsRegistry = $service->get();
        $this->register($pluginsRegistry, $application);
    }
    
    
    /**
     * Registers all plugins in the application.
     *
     * @param array       $pluginsRegistry
     * @param Application $application
     */
    private function register(array $pluginsRegistry, Application $application): void
    {
        /**
         * @var SlimApp                $slim
         * @var EventListenerProvider  $listenerProvider
         * @var CommandHandlerProvider $handlerProvider
         * @var LoggerInterface        $logger
         */
        $slim             = $application->get(SlimApp::class);
        $listenerProvider = $application->get(EventListenerProvider::class);
        $handlerProvider  = $application->get(CommandHandlerProvider::class);
        $logger           = $application->get(LoggerBuilder::class)->build();
        
        foreach ($pluginsRegistry as $entryPoint => $className) {
            try {
                if (!class_exists($className) && file_exists($entryPoint)) {
                    /** @noinspection PhpIncludeInspection */
                    require_once $entryPoint;
                }
                
                /** @var Plugin $plugin */
                $plugin = $application->has($className) ? $application->get($className) : new $className;
                
                $this->registerServiceProviders($plugin, $application);
                $this->registerCommandHandlers($plugin, $handlerProvider);
                $this->registerEventListeners($plugin, $listenerProvider);
                $this->registerGetRoutes($plugin, $slim);
                $this->registerPostRoutes($plugin, $slim);
                $this->registerPutRoutes($plugin, $slim);
                $this->registerDeleteRoutes($plugin, $slim);
            } catch (Throwable $e) {
                $msg     = "Failed to load plugin ({$className})";
                $context = [
                    'classname'   => $className,
                    'entry-point' => $entryPoint,
                    'message'     => $e->getMessage()
                ];
                
                $logger->error($msg, $context);
            }
        }
    }
    
    
    /**
     * Registers service providers to the application.
     *
     * @param Plugin      $plugin
     * @param Application $application
     */
    private function registerServiceProviders(Plugin $plugin, Application $application): void
    {
        foreach ($plugin->serviceProviders() ?? [] as $serviceProvider) {
            /** @var PluginServiceProvider $provider */
            $provider = new $serviceProvider($application);
            $application->addServiceProvider($provider->toLeagueInterface());
        }
    }
    
    
    /**
     * Registers event listeners to the application.
     *
     * @param Plugin                $plugin
     * @param EventListenerProvider $listenerProvider
     */
    private function registerEventListeners(Plugin $plugin, EventListenerProvider $listenerProvider): void
    {
        foreach ($plugin->eventListeners() ?? [] as $event => $listeners) {
            foreach ($listeners as $listener) {
                $listenerProvider->attachListener($event, $listener);
            }
        }
    }
    
    
    /**
     * Registers command handlers to the application.
     *
     * @param Plugin                 $plugin
     * @param CommandHandlerProvider $handlerProvider
     */
    private function registerCommandHandlers(Plugin $plugin, CommandHandlerProvider $handlerProvider): void
    {
        foreach ($plugin->commandHandlers() ?? [] as $command => $handlers) {
            foreach ($handlers as $handler) {
                $handlerProvider->attachHandler($command, $handler);
            }
        }
    }
    
    
    /**
     * Registers get routes to the application.
     *
     * @param Plugin  $plugin
     * @param SlimApp $slim
     */
    private function registerGetRoutes(Plugin $plugin, SlimApp $slim): void
    {
        foreach ($plugin->getRoutes() ?? [] as $route => $mapping) {
            $callback = implode(':', $mapping);
            $slim->get($route, $callback);
        }
    }
    
    
    /**
     * Registers post routes to the application.
     *
     * @param Plugin  $plugin
     * @param SlimApp $slim
     */
    private function registerPostRoutes(Plugin $plugin, SlimApp $slim): void
    {
        foreach ($plugin->postRoutes() ?? [] as $route => $mapping) {
            $callback = implode(':', $mapping);
            $slim->post($route, $callback);
        }
    }
    
    
    /**
     * Registers put routes to the application.
     *
     * @param Plugin  $plugin
     * @param SlimApp $slim
     */
    private function registerPutRoutes(Plugin $plugin, SlimApp $slim): void
    {
        foreach ($plugin->putRoutes() ?? [] as $route => $mapping) {
            $callback = implode(':', $mapping);
            $slim->put($route, $callback);
        }
    }
    
    
    /**
     * Registers delete routes to the application.
     *
     * @param Plugin  $plugin
     * @param SlimApp $slim
     */
    private function registerDeleteRoutes(Plugin $plugin, SlimApp $slim): void
    {
        foreach ($plugin->deleteRoutes() ?? [] as $route => $mapping) {
            $callback = implode(':', $mapping);
            $slim->delete($route, $callback);
        }
    }
    
    
    /**
     * Provides the plugins service from the applications di container.
     *
     * @param Application $application
     *
     * @return PluginRegistryService
     */
    private function getPluginsService(Application $application): PluginRegistryService
    {
        return $application->get(PluginRegistryService::class);
    }
}